#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.VbecsUser;

#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_VbecsUser : BaseTest 
	{
		
		private DataTable _dtUser;
		private DataTable _dtUserDivRole;
		private BOL.VbecsUser _testVbecsUser;
		private string _testVbecsUserId;

		[SetUp]
		protected void SetUp() 
		{				
			this.BaseSetUp();

			if (this.RefreshData)
			{
				_dtUser = UnitTestUtil.RunSQLGetDataTable( "SELECT * FROM VbecsUser WHERE VbecsUserId = '" + COMMON.LogonUser.LogonUserName + "'", true);
				_dtUserDivRole = UnitTestUtil.RunSQLGetDataTable( "SELECT TOP 1 * FROM VbecsUserDivisionRole WHERE VbecsUserId = '" + COMMON.LogonUser.LogonUserName + "' AND RecordStatusCode = 'A'", true);
				
				this.RefreshData = false;
			}

			_testVbecsUserId = this._dtUser.Rows[0][ TABLES.VbecsUser.VbecsUserId ].ToString();

			_testVbecsUser = new BOL.VbecsUser( _testVbecsUserId );
			_testVbecsUser.LogonToDivision( UnitTestConfig.TestDivisionCode );
		}

		[Test]
		public void Ctor_Pass()
		{
			BOL.VbecsUser _testObj = new VbecsUser();

			Assert.IsTrue( _testObj.VbecsUserGuid != Guid.Empty );
			Assert.IsNull( _testObj.UserNTLoginID );
			Assert.IsNull( _testObj.UserDuz );
			Assert.IsNull( _testObj.UserName );
			Assert.IsNull( _testObj.UserInitials );
			Assert.IsNull( _testObj.VistAUserName );
			Assert.IsNull( _testObj.UserEmail );

			Assert.IsTrue( _testObj.IsActive );
			Assert.IsTrue( _testObj.IsDirty );
			Assert.IsTrue( _testObj.IsNew );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Fail() {}

		[Test]
		public void Ctor_String_Bool_Pass()
		{
			BOL.VbecsUser _testUser = new BOL.VbecsUser( _testVbecsUserId, true );

			Assert.IsNotNull( _testUser );
			Assert.AreEqual( _testVbecsUserId, _testUser.UserNTLoginID );
		}

		[Test]
		[ExpectedException( typeof(ArgumentNullException) )]
		public void Ctor_String_Bool_Fail() 
		{
			new VbecsUser( null );
		}

		[Test]
		public void Ctor_String_Pass()
		{
			BOL.VbecsUser _testUser = new BOL.VbecsUser( _testVbecsUserId );

			Assert.IsNotNull( _testUser );
			Assert.AreEqual( _testVbecsUserId, _testUser.UserNTLoginID );
		}

		[Test]
		[ExpectedException( typeof(ArgumentNullException) )]
		public void Ctor_String_Fail() 
		{
			new VbecsUser( null );
		}

		[Test]
		public void RefreshFromDb_Pass()
		{
			this.RefreshData = true;

			string _probeEmail = _testVbecsUser.UserEmail;

			_testVbecsUser.UserEmail = "abc" + _probeEmail;
			Assert.IsTrue( _testVbecsUser.IsDirty );

			_testVbecsUser.RefreshFromDb();

			Assert.IsTrue( !_testVbecsUser.IsDirty );
			Assert.AreEqual( _probeEmail, _testVbecsUser.UserEmail );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void RefreshFromDb_Fail() {}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			this.RefreshData = true;

			//Emails are optional and sometimes blank, so we give it a test value
			_testVbecsUser.UserEmail = "test@test.com";

			DataRow _dr = _dtUser.NewRow();
			_testVbecsUser.LoadDataRowFromThis( _dr );

			Assert.AreEqual( _testVbecsUser.VbecsUserGuid, (Guid)_dr[ TABLES.VbecsUser.VbecsUserGuid ] );
			Assert.AreEqual( _testVbecsUser.UserNTLoginID, _dr[ TABLES.VbecsUser.VbecsUserId ].ToString() );			
			Assert.AreEqual( _testVbecsUser.UserName, _dr[ TABLES.VbecsUser.UserName ].ToString() );
			Assert.AreEqual( _testVbecsUser.UserDuz, _dr[ TABLES.VbecsUser.UserDUZ ].ToString() );
			Assert.AreEqual( _testVbecsUser.VistAUserName, _dr[ TABLES.VbecsUser.VistaUserName ].ToString() );			
			Assert.AreEqual( _testVbecsUser.UserEmail, _dr[ TABLES.VbecsUser.UserEmail ].ToString() );
			Assert.AreEqual( _testVbecsUser.UserInitials, _dr[ TABLES.VbecsUser.UserInitials ].ToString() );
			Assert.AreEqual( _testVbecsUser.IsActive, _dr[ TABLES.VbecsUser.RecordStatusCode ].ToString() == "A" );
		}	

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void LoadDataRowFromThis_Fail_1()
		{		
			_testVbecsUser.LoadDataRowFromThis( new DataTable().NewRow() );
		}

		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void LoadDataRowFromThis_Fail_2()
		{		
			_testVbecsUser.LoadDataRowFromThis( null );
		}

		[Test]
		[ExpectedException(typeof(BusinessObjectException))]
		public void LoadDataRowFromThis_Fail_3()
		{		
			_testVbecsUser.UserEmail = "...";
			Assert.IsTrue( !_testVbecsUser.IsValid );

			_testVbecsUser.LoadDataRowFromThis( _dtUser.NewRow() );
		}

		[Test]
		public void IsAuthorizedForFunction_Pass_1()
		{
			Assert.IsTrue( _testVbecsUser.IsAuthorizedForFunction(Common.FunctionCode.StandardAccess ) );
		}

		[Test]
		public void IsAuthorizedForFunction_Pass_2()
		{
			Assert.IsTrue( !this.GetLoggedOnTechnologistUser().IsAuthorizedForFunction( Common.FunctionCode.DefineVBECSUsers ) );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsAuthorizedForFunction_Fail() {}

		[Test]
		public void IsAuthorized_Pass_1()
		{
			Assert.IsTrue( this.GetLoggedOnTechnologistUser().IsAuthorized( Common.UserRoleType.Technologist ) );
		}

		[Test]
		public void IsAuthorized_Pass_2()
		{
			Assert.IsTrue( !this.GetLoggedOnTechnologistUser().IsAuthorized( Common.UserRoleType.AdministratorSupervisor ) );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsAuthorized_Fail()
		{
		}

		private VbecsUser GetLoggedOnTechnologistUser()
		{
			BOL.VbecsUser techUser = this.TechUser;

			techUser.LogonToDivision( UnitTestConfig.TestDivisionCode );
			
			return techUser;
		}


		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void LogonToVbecs_Pass()
		{
			// It is unjustifiably time-consuming to set up a test Windows group.
		}

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void LogonToVbecs_Fail()
		{
		}

		[Test]
		public void LogonToDivision_Pass()
		{
			this.RefreshData = true;
			
			VbecsUser _tempTestUser = GetTempUserLoggedOnToUnitTestDivision();

			Assert.AreEqual( UnitTestConfig.TestDivisionCode.Trim().ToUpper(), _tempTestUser.CurrentDivisionRole.Division.DivisionCode );
			Assert.IsTrue( _tempTestUser.CurrentDivisionRole.IsActive );
			Assert.AreEqual( _tempTestUser, VbecsUser.LoggedOnUser );

			Assert.AreEqual( _tempTestUser.CurrentDivisionRole.Division.DivisionCode, Common.LogonUser.LogonUserDivisionCode );
			Assert.AreEqual( _tempTestUser.UserDuz, Common.LogonUser.LogonUserDUZ );
		}

		[Test]
		[ExpectedException( typeof(NullReferenceException ) )]
		public void LogonToDivision_Fail()
		{
			this.RefreshData = true;
	
			_testVbecsUser.LogonToDivision( String.Empty );
		}

		[Test]
		public void LoggedOnUser_Pass()
		{
			this.RefreshData = true;

			VbecsUser _tempTestUser = GetTempUserLoggedOnToUnitTestDivision();

			Assert.AreEqual( _tempTestUser, VbecsUser.LoggedOnUser );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.ReadOnlyProperty )]
		public void LoggedOnUser_Fail() {}

		private VbecsUser GetTempUserLoggedOnToUnitTestDivision()
		{
			VbecsUser _tempTestUser = this.TestUser;
			_tempTestUser.LogonToDivision( UnitTestConfig.TestDivisionCode );
			
			return _tempTestUser;
		}

		[Test]
		public void GetActiveDivisionUsers_Pass()
		{
			IList _testList = VbecsUser.GetActiveDivisionUsers( UnitTestConfig.TestDivisionCode );

			Assert.IsNotNull( _testList );
			Assert.IsTrue( _testList.Count > 0 );
			
			foreach( VbecsUser _user in _testList )
			{
				Assert.IsTrue( _user.IsActive );
				Assert.IsNotNull( _user.DivisionsAndRoles[ UnitTestConfig.TestDivisionCode ] );
				Assert.IsTrue( _user.DivisionsAndRoles[ UnitTestConfig.TestDivisionCode ].IsActive );
			}
		}

		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void GetActiveDivisionUsers_Fail_1() 
		{
			VbecsUser.GetActiveDivisionUsers( null );
		}

		[Test]
		public void GetActiveDivisionUsers_Fail_2() 
		{
			Assert.AreEqual( 0, VbecsUser.GetActiveDivisionUsers( "00000" ).Count );
		}

		[Test]
		public void GetAllUsers_Pass()
		{
			IList _testList = VbecsUser.GetAllUsers();

			Assert.IsNotNull( _testList );
			Assert.IsTrue( _testList.Count > 0 );

			foreach( object _obj in _testList )
				Assert.IsTrue( _obj is VbecsUser );
		}	

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetAllUsers_Fail() {}

		[Test]
		public void GetObjectID_Pass()
		{
			Assert.AreEqual( _testVbecsUserId, _testVbecsUser.GetObjectID() );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetObjectID_Fail() {}

		[Test]
		public void GetAllSortedById_Pass()
		{
			BusinessObjectSortedHashList _testList = VbecsUser.GetAllSortedById();
			
			Assert.IsNotNull( _testList );
			Assert.IsTrue( _testList.Count > 0 );
			Assert.AreEqual( _testVbecsUser.VbecsUserGuid, ((VbecsUser)_testList.GetByObjectID( _testVbecsUserId )).VbecsUserGuid );			
			
			foreach( object _obj in _testList )
				Assert.IsTrue( _obj is VbecsUser );

			VbecsUser _prevUser = null;

			foreach( VbecsUser _user in _testList )
			{			
				if( _prevUser != null )
					Assert.IsTrue( _user.UserNTLoginID.CompareTo( _prevUser.UserNTLoginID ) >= 0 );

				_prevUser = _user;
			}
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetAllSortedById_Fail() {}

		[Test]
		public void InactivateUsers_Pass() 
		{
			this.RefreshData = true;

			UnitTestUtil.RunSQL("UPDATE VbecsUser SET RecordStatusCode = 'A' WHERE VbecsUserId <> '" + Common.LogonUser.LogonUserName + "'");
			UnitTestUtil.RunSQL("UPDATE VbecsUserDivisionRole SET RecordStatusCode = 'A' WHERE VbecsUserId <> '" + Common.LogonUser.LogonUserName + "'");

			BOL.VbecsUser tmpUser = new BOL.VbecsUser("VHAISHVBECS1");	//Cheating, but feelin too lazy to code around "VBECS" user snafu

			System.Collections.ArrayList tmpList = new ArrayList();
			tmpList.Add(tmpUser);

			BOL.VbecsUser.InactivateUsers(tmpList, Common.UpdateFunction.UnitTests);
			Assert.IsTrue(true, "No return value");

			//Data Check
			DataTable dtUsers = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM VbecsUser WHERE VbecsUserId = '" + tmpUser.UserNTLoginID + "'");
			Assert.AreEqual("X", dtUsers.Rows[0][TABLES.VbecsUser.RecordStatusCode].ToString(), "Data Check");

			//Return to the earth, the egg we took from the suitcase
			UnitTestUtil.RunSQL("UPDATE VbecsUser SET RecordStatusCode = 'A' WHERE VbecsUserId = '" + tmpUser.UserNTLoginID + "'");
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void InactivateUsers_Fail() 
		{
			this.RefreshData = true;

			DataTable dtUsers = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM VbecsUser WHERE VbecsUserId <> '" + Common.LogonUser.LogonUserName + "'");

			BOL.VbecsUser tmpUser = new BOL.VbecsUser(dtUsers.Rows[0][TABLES.VbecsUser.VbecsUserId].ToString());

			System.Collections.ArrayList tmpList = new ArrayList();
			tmpList.Add(tmpUser);
			tmpList.Add(new BOL.VbecsUser());

			BOL.VbecsUser.InactivateUsers(tmpList, Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void CurrentDivisionRole_Pass()
		{
			VbecsUser _testUser = GetTempUserLoggedOnToUnitTestDivision();

			Assert.IsNotNull( _testUser.CurrentDivisionRole );
			Assert.AreEqual( UnitTestConfig.TestDivisionCode.Trim().ToUpper(), _testUser.CurrentDivisionRole.Division.DivisionCode );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.ReadOnlyProperty )]
		public void CurrentDivisionRole_Fail() {}

		[Test]
		public void DivisionsAndRoles_Pass()
		{
			Assert.IsNotNull( _testVbecsUser.DivisionsAndRoles );
			Assert.IsTrue( _testVbecsUser.DivisionsAndRoles.Count > 0 );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.ReadOnlyProperty )]
		public void DivisionsAndRoles_Fail() {}

		[Test]
		public void VbecsUserGuid_Pass()
		{
			Assert.IsTrue( _testVbecsUser.VbecsUserGuid != Guid.Empty );			
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void VbecsUserGuid_Fail() {}

		[Test]
		public void UserNTLoginID_Pass()
		{
			this.RefreshData = true;

			string _testLoginId = "AAAAA" + UnitTestUtil.GetRandomString( 20 ); // user NT login ID can't be empty
			_testVbecsUser.UserNTLoginID = _testLoginId;

			Assert.AreEqual( _testLoginId, _testVbecsUser.UserNTLoginID );

			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserNtLoginIdNotSet ) );
			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserNtLoginIdInvalid ) );
			Assert.IsTrue( _testVbecsUser.IsDirty );
		}

		[Test]
		public void UserNTLoginID_Fail_1() 
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			string _testString = "A".PadRight( 100, 'B' );
			_testVbecsUser.UserNTLoginID = _testString;

			Assert.AreEqual( _testString, _testVbecsUser.UserNTLoginID );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserNtLoginIdInvalid, _beforeUpdateBrokenRulesCount );
		}
		
		[Test]
		public void UserNTLoginID_Fail_2()
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			_testVbecsUser.UserNTLoginID = null;

			Assert.IsNull( _testVbecsUser.UserNTLoginID );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserNtLoginIdNotSet, _beforeUpdateBrokenRulesCount );
		}		

		[Test]
		public void UserName_Pass()
		{
			this.RefreshData = true;

			string _testName = UnitTestUtil.GetRandomString( 50 );
			_testVbecsUser.UserName = _testName;
			
			Assert.AreEqual( _testName, _testVbecsUser.UserName );

			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserNtNameNotSet ) );
			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserNameTooLong ) );
			Assert.IsTrue( _testVbecsUser.IsDirty );
		}

		[Test]
		public void UserName_Fail_1() 
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			string _testString = "A".PadRight( 100, 'B' );
			_testVbecsUser.UserName = _testString;

			Assert.AreEqual( _testString, _testVbecsUser.UserName );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserNameTooLong, _beforeUpdateBrokenRulesCount );
		}
		
		[Test]
		public void UserName_Fail_2()
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			_testVbecsUser.UserName = null;

			Assert.IsNull( _testVbecsUser.UserName );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserNtNameNotSet, _beforeUpdateBrokenRulesCount );
		}

		[Test]
		public void UserDuz_Pass()
		{
			this.RefreshData = true;

			string _testDuz = UnitTestUtil.GetRandomUniqueUserDuz();
			_testVbecsUser.UserDuz = _testDuz;
			
			Assert.AreEqual( _testDuz, _testVbecsUser.UserDuz );

			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserDuzNotSet ) );
			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserDuzInvalid ) );
			Assert.IsTrue( _testVbecsUser.IsDirty );
		}

		[Test]
		public void UserDuz_Fail_1() 
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			string _testString = "1".PadRight( 20, '0' );
			_testVbecsUser.UserDuz = _testString;

			Assert.AreEqual( _testString, _testVbecsUser.UserDuz );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserDuzInvalid, _beforeUpdateBrokenRulesCount );
		}
		
		[Test]
		public void UserDuz_Fail_2()
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			_testVbecsUser.UserDuz = null;

			Assert.IsNull( _testVbecsUser.UserDuz );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserDuzNotSet, _beforeUpdateBrokenRulesCount );
		}

		[Test]
		public void VistAUserName_Pass()
		{
			this.RefreshData = true;

			string _testVistaName = UnitTestUtil.GetRandomString( 50 );
			_testVbecsUser.VistAUserName = _testVistaName;
			
			Assert.AreEqual( _testVistaName, _testVbecsUser.VistAUserName );

			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.VistaUserNameNotSet ) );
			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.VistaUserNameTooLong ) );
			Assert.IsTrue( _testVbecsUser.IsDirty );
		}

		[Test]
		public void VistAUserName_Fail_1() 
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			string _testString = "A".PadRight( 100, 'B' );
			_testVbecsUser.VistAUserName = _testString;

			Assert.AreEqual( _testString, _testVbecsUser.VistAUserName );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.VistaUserNameTooLong, _beforeUpdateBrokenRulesCount );
		}
		
		[Test]
		public void VistAUserName_Fail_2()
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			_testVbecsUser.VistAUserName = null;

			Assert.IsNull( _testVbecsUser.VistAUserName );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.VistaUserNameNotSet, _beforeUpdateBrokenRulesCount );
		}
		
		[Test]
		public void UserEmail_Pass()
		{
			this.RefreshData = true;

			string _testEmail = UnitTestUtil.GetRandomString( 10 ) + "@domain.ext";
			_testVbecsUser.UserEmail = _testEmail;

			Assert.AreEqual( _testEmail, _testVbecsUser.UserEmail );
			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserEmailInvalid ) );
			Assert.IsTrue( _testVbecsUser.IsDirty );
		}

		[Test]
		public void UserEmail_Fail()
		{
			this.RefreshData = true;
			
			string _testString = UnitTestUtil.GetRandomString( 10 );
			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			_testVbecsUser.UserEmail = _testString;			

			Assert.AreEqual( _testString, _testVbecsUser.UserEmail );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserEmailInvalid, _beforeUpdateBrokenRulesCount );
		}

		[Test]
		public void UserInitials_Pass()
		{
			this.RefreshData = true;

			string _testInitials = UnitTestUtil.GetRandomString( 3 );
			_testVbecsUser.UserInitials = _testInitials;
			
			Assert.AreEqual( _testInitials, _testVbecsUser.UserInitials );

			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserInitialsNotSet ) );
			Assert.IsTrue( !_testVbecsUser.IsRuleBroken( BROKEN.UserInitialsAreTooLong ) );
			Assert.IsTrue( _testVbecsUser.IsDirty );
		}

		[Test]
		public void UserInitials_Fail_1() 
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			string _testString = "AAAAAAA";
			_testVbecsUser.UserInitials = _testString;

			Assert.AreEqual( _testString, _testVbecsUser.UserInitials );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserInitialsAreTooLong, _beforeUpdateBrokenRulesCount );
		}
		
		[Test]
		public void UserInitials_Fail_2()
		{
			this.RefreshData = true;

			int _beforeUpdateBrokenRulesCount = _testVbecsUser.BrokenRulesCount;
			_testVbecsUser.UserInitials = null;

			Assert.IsNull( _testVbecsUser.UserInitials );
			VerifySetterPropertyBreakingRuleForTestUser( BROKEN.UserInitialsNotSet, _beforeUpdateBrokenRulesCount );
		}

		private void VerifySetterPropertyBreakingRuleForTestUser( string brokenRuleName, int beforeBrokenRuleCount )
		{
			Assert.AreEqual( beforeBrokenRuleCount + 1, _testVbecsUser.BrokenRulesCount );
			Assert.IsTrue( _testVbecsUser.IsRuleBroken( brokenRuleName ) );
			Assert.IsTrue( _testVbecsUser.IsDirty );
		}

		[Test]
		public void IsActive_Pass_1()
		{
			VerifyTestIsActiveOnTestUser( false );
		}

		[Test]
		public void IsActive_Pass_2()
		{
			VerifyTestIsActiveOnTestUser( true );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void IsActive_Fail() {}

		private void VerifyTestIsActiveOnTestUser( bool isActive )
		{
			_testVbecsUser.IsActive = !isActive; // resetting the object for the test

			_testVbecsUser.IsActive = isActive;

			Assert.AreEqual( isActive, _testVbecsUser.IsActive );
			Assert.IsTrue( _testVbecsUser.IsDirty );
		}

		[Test]
		public void GetUserRolesList_Pass()
		{
			DataTable dtTest = BOL.VbecsUser.GetUserRolesList((int)Common.UserRoleType.AdministratorSupervisor);

			Assert.IsTrue( dtTest.Rows.Count > 0);
			Assert.IsTrue( dtTest.Columns.Count == 2);
			Assert.IsTrue( dtTest.Columns.Contains(TABLES.UserRole.UserRoleId) );
			Assert.IsTrue( dtTest.Columns.Contains(TABLES.UserRole.UserRoleName) );
		}

		[Test]
		public void GetUserRolesList_Fail()
		{
			DataTable dtTest = BOL.VbecsUser.GetUserRolesList( -8 );
			Assert.AreEqual( 0, dtTest.Rows.Count );
		}

		[Test]
		public void Persist_Pass()
		{
			this._testVbecsUser.Persist(Common.UpdateFunction.UnitTests);
			Assert.IsTrue(true, "No return value");
		}

		[Test]
		[ExpectedException(typeof(Common.RowVersionException))]
		public void Persist_Fail()
		{
			this._testVbecsUser.RowVersion = new byte[8];
			this._testVbecsUser.UserNTLoginID = "xxx";
		
			this._testVbecsUser.Persist(Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void PersistUsers_Pass()
		{
			System.Collections.ArrayList users = new System.Collections.ArrayList(1);
			users.Add(_testVbecsUser);
			//
			BOL.VbecsUser.PersistUsers(users, Common.UpdateFunction.UnitTests);
			//If no exception was thrown, it succeeded.
			Assert.IsTrue(true);
		}

		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void PersistUsers_Fail()
		{
			BOL.VbecsUser.PersistUsers(null, Common.UpdateFunction.UnitTests);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void LogonToVbecsAdministrator_Pass()
		{
			// It is unjustifiably time-consuming to set up a test Windows group.
		}

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void LogonToVbecsAdministrator_Fail()
		{
			// It is unjustifiably time-consuming to set up a test deleting and re-creating Windows group.
		}

		[Test]
		public void GetVbecsUsersWithRoleInGivenDivision_Pass()
		{
			System.Collections.IList users = BOL.VbecsUser.GetVbecsUsersWithRoleInGivenDivision(UnitTestConfig.TestDivisionCode);
			Assert.IsTrue(users.Count>0,"UsersInDivision");
		}

		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void GetVbecsUsersWithRoleInGivenDivision_Fail()
		{
			System.Collections.IList users = BOL.VbecsUser.GetVbecsUsersWithRoleInGivenDivision(null);
			Assert.IsNull(users);
		}

		[Test]
		public void IsDirty_Pass()
		{
			_testVbecsUser.IsDirty = true;
			Assert.AreEqual(true,_testVbecsUser.IsDirty,"IsDirty");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IsDirty_Fail()
		{
		}

		[Test]
		public void SetDefaultPrinter_Pass()
		{
			// Get all printers installed on server
			string [] printers = (string []) gov.va.med.vbecs.Common.Utility.GetPrintersInstalledOnServer();	
			// Save valid printer to DB
			string sqlUpdate = "UPDATE VamcDivision SET PrinterName = '" + printers[0] + "'";
			UnitTestUtil.RunSQL(sqlUpdate);
			
			// recreate to get new data
			_testVbecsUser = new BOL.VbecsUser( _testVbecsUserId );
			_testVbecsUser.LogonToDivision( UnitTestConfig.TestDivisionCode );
			Assert.IsTrue(_testVbecsUser.SetDefaultPrinter());
		}
		[Test]
		public void SetDefaultPrinter_Fail()
		{
			// Save invalid printer to DB
			string sqlUpdate = "UPDATE VamcDivision SET PrinterName = 'NoPrinterWithThisName'";
			UnitTestUtil.RunSQL(sqlUpdate);

			// recreate to get new data
			_testVbecsUser = new BOL.VbecsUser( _testVbecsUserId );
			_testVbecsUser.LogonToDivision( UnitTestConfig.TestDivisionCode );
			Assert.IsTrue(!_testVbecsUser.SetDefaultPrinter());
		}
	}
}
#endif
